import pygame
from utils import spritesheet
import random

class Food(pygame.sprite.Sprite):
    #private constants
    _DEFAULT_SIZE = [32, 32]

    #images
    _IMG_FACE_NORTH = (128, 96, 32, 32)
    _IMG_FACE_EAST = (128, 64, 32, 32)
    _IMG_FACE_WEST = (128, 32, 32, 32)
    _IMG_FACE_SOUTH = (128, 0, 32, 32)


    def __init__(self, position):
        #parameter validation
        size = Food._DEFAULT_SIZE
        if position == None:
            raise Exception('Invalid position.')

        self.filename = 'images/Animal.png'
        ss = spritesheet.Spritesheet('images/Animal.png')
        
        #initialization
        randompos = random.choice([self._IMG_FACE_NORTH, self._IMG_FACE_EAST, self._IMG_FACE_WEST, self._IMG_FACE_SOUTH])
        self.size = size
        self.image = ss.image_at(randompos, colorkey=-1)
        self.rect = self.image.get_rect()
        self.rect.topleft = position
        

class Snake(pygame.sprite.Sprite):
    #private constants
    _DEFAULT_SIZE = [32, 32]
    _DEFAULT_POSITION = [96, 96] #space given to tail of length 2

    #images
    _HOTSPOTS = {
        'north': [(32, 96, 32, 32), (0, 96, 32, 32), (64, 96, 32, 32)],
        'east': [(32, 64, 32, 32), (0, 64, 32, 32), (64, 64, 32, 32)],
        'west': [(32, 32, 32, 32), (0, 32, 32, 32), (64, 32, 32, 32)],
        'south': [(32, 0, 32, 32), (0, 0, 32, 32), (64, 0, 32, 32)]
    }

    #-----PRIVATE CLASSES-------------------------------------------------------
    class _SnakeTail(pygame.sprite.Sprite):
        def __init__(self, spritesheet):
            #initialization
            self.tiles = []
            self.ss = spritesheet

        def add_tile(self, position, direction):
            #creates a new tile
            pos, posarr = Snake.get_sprite_coords_for_direction(direction)
            tile = self.ss.image_at(pos, colorkey=-1)
            rect = tile.get_rect()
            rect.topleft = position

            self.tiles.append({'image':tile, 'rect':rect})

        def update_image(self, rects):
            for i in range(len(self.tiles)):
                self.tiles[i]['image'] = self.ss.image_at(random.choice(rects), colorkey=-1)

    class _SnakeHead(pygame.sprite.Sprite):
        #images
        _HOTSPOTS = {
            'north': [(224, 224, 32, 32), (192, 224, 32, 32), (256, 224, 32, 32)],
            'east': [(224, 192, 32, 32), (192, 192, 32, 32), (256, 192, 32, 32)],
            'west': [(224, 160, 32, 32), (192, 160, 32, 32), (256, 160, 32, 32)],
            'south': [(224, 128, 32, 32), (192, 128, 32, 32), (256, 128, 32, 32)]
        }

        def __init__(self, position, direction, spritesheet):
            #initialization
            pos, posarr = Snake.get_sprite_coords_for_direction(direction, Snake._SnakeHead._HOTSPOTS)
            self.ss = spritesheet
            self.image = self.ss.image_at(pos, colorkey=-1)
            self.rect = self.image.get_rect()
            self.rect.topleft = position

        def update_image(self, rect):
            self.image = self.ss.image_at(rect, colorkey=-1)

    #---------------------------------------------------------------------------

    class SnakeMove():
        UP = '1Y'
        DOWN = '-1Y'
        RIGHT = '1X'
        LEFT = '-1X'

        DIRECTIONS = { UP: 'north', DOWN: 'south', RIGHT: 'east', LEFT: 'west' }

        #checks a direction's validity
        @staticmethod
        def SnakeMove_is_member(direction):
            try:
                idx = [Snake.SnakeMove.UP, Snake.SnakeMove.DOWN, Snake.SnakeMove.RIGHT, Snake.SnakeMove.LEFT].index(direction)
                return True
            except:
                return False

    def __init__(self, position):
        if position == None:
            position = Snake._DEFAULT_POSITION

        size = Snake._DEFAULT_SIZE

        self.filename = 'images/Animal.png'
        self.ss = spritesheet.Spritesheet('images/Animal.png')
        self.direction = Snake.SnakeMove.RIGHT

        self.size = size
        self.head = Snake._SnakeHead(position, self.direction, self.ss)
        self.tail = Snake._SnakeTail(self.ss)
        
        tailposition = [(position[0] - size[0]), position[1]]
        self.tail.add_tile(tailposition, Snake.SnakeMove.RIGHT)
        tailposition = [(position[0] - 2*size[0]), position[1]]
        self.tail.add_tile(tailposition, Snake.SnakeMove.RIGHT)

    def move(self, direction, frame_width, frame_height, barrierposarr):
        #parameter validation
        if Snake.SnakeMove.SnakeMove_is_member(direction) != True:
            raise Exception('Invalid movement direction.')

        #initializes new position
        self.direction = direction
        newheadposition = [self.head.rect.topleft[0], self.head.rect.topleft[1]]
        newheadposition = self.get_adjusted_pos(newheadposition, frame_width, frame_height)
        
        self.update_sprite()

        if self.did_collide(newheadposition, barrierposarr):
            return False

        if self.occupies_position(newheadposition):
            return False

        #moves the head to its new position
        self.move_tail(self.head.rect.topleft)
        self.head.rect.topleft = newheadposition

        return True

    def move_tail(self, newtileposition):
        for count in range(len(self.tail.tiles)):
            prevtileposition = self.tail.tiles[count]['rect'].topleft
            self.tail.tiles[count]['rect'].topleft = newtileposition
            newtileposition = prevtileposition         

    def get_adjusted_pos(self, headpos, frame_width, frame_height):
        stepsize = self.head.image.get_rect()[2]
        if self.direction == Snake.SnakeMove.UP:
            headpos[1] = (headpos[1]-stepsize)%frame_height
        if self.direction  == Snake.SnakeMove.DOWN:
            headpos[1] = (headpos[1]+stepsize)%frame_height
        if self.direction == Snake.SnakeMove.RIGHT:
            headpos[0] = (headpos[0]+stepsize)%frame_width
        if self.direction == Snake.SnakeMove.LEFT:
            headpos[0] = (headpos[0]-stepsize)%frame_width

        return headpos

    def did_collide(self, newposs, posarr):
        try:
            idx = posarr.index(newposs)
            return True
        except:
            return False

    def update_sprite(self):
        hpos, hposarr = Snake.get_sprite_coords_for_direction(self.direction, Snake._SnakeHead._HOTSPOTS)
        tpos, tposarr = Snake.get_sprite_coords_for_direction(self.direction)
            
        self.head.update_image(hpos)
        self.tail.update_image(tposarr)

    @staticmethod
    def get_sprite_coords_for_direction(direction, hotspots=None):
        if hotspots is None:
            hotspots = Snake._HOTSPOTS
        key = Snake.SnakeMove.DIRECTIONS[direction]
        pos = random.choice(hotspots[key])
        posarr = hotspots[key]

        return pos, posarr


    #checks if this snake's body occupies a given position
    def occupies_position(self, position):
        #parameter validation
        if position[0] == None or position[1] == None or self.head.rect.topleft == position:
            return True

        occupied_pos = [tile['rect'].topleft for tile in self.tail.tiles if tile['rect'].topleft == position]
        if len(occupied_pos) > 0:
            return True

        return False

    def lengthen_tail(self, number=1):
        size = self.size[0]

        for count in range(number):
            lastindex = len(self.tail.tiles) - 1
            X = self.tail.tiles[lastindex]['rect'].topleft[0]
            Y = self.tail.tiles[lastindex]['rect'].topleft[1]

            #determines position of new tile
            if self.direction == Snake.SnakeMove.UP:
                Y = Y - size + (count*size)
            elif self.direction == Snake.SnakeMove.DOWN:
                Y = Y + size + (count*size)
            elif self.direction == Snake.SnakeMove.RIGHT:
                X = X - size + (count*size)
            elif self.direction == Snake.SnakeMove.LEFT:
                X = X + size + (count*size)

            self.tail.add_tile([X, Y], self.direction)